<?php
// Configuração de relatórios de erros e log de eventos
error_reporting(E_ALL);
ini_set('log_errors', 1);
ini_set('error_log', 'webhook_errors.log');

require 'config.php'; // Usa a conexão existente no projeto

// Função para salvar logs personalizados
function custom_log($message, $type = 'info') {
    $log_file = __DIR__ . '/custom_webhook_logs.log'; // Caminho do arquivo de log
    $log_message = "[" . date('Y-m-d H:i:s') . "] [$type] - $message\n";
    file_put_contents($log_file, $log_message, FILE_APPEND);
}

// Função para processar os eventos da Hotmart
function process_event($conn, $event_type, $data) {
    custom_log("Iniciando processamento do evento: $event_type", 'info');
    
    switch ($event_type) {
        case 'COMPRA_CANCELADA':
        case 'PURCHASE_CANCELED':
            custom_log("Evento: Compra Cancelada", 'result');
            update_subscription_status($conn, $data, 'cancelada', true, 0, 'Cancelada'); // Define status_assinatura para "Cancelada"
            break;

        case 'COMPRA_COMPLETA':
        case 'PURCHASE_COMPLETE':
            custom_log("Evento: Compra Completa", 'result');
            save_user_and_subscription($conn, $data, 30); // Atualiza dias_restantes para 30 dias
            update_subscription_status($conn, $data, 'completa', false, 30, 'Ativa'); // Define status_assinatura para "Ativa"
            break;

        case 'COMPRA_APROVADA':
        case 'PURCHASE_APPROVED':
            custom_log("Evento: Compra Aprovada", 'result');
            save_user_and_subscription($conn, $data, 30); // Atualiza dias_restantes para 30 dias
            update_subscription_status($conn, $data, 'aprovada', false, 30, 'Ativa'); // Define status_assinatura para "Ativa"
            break;

        case 'PURCHASE_CHARGEBACK':
        case 'PURCHASE_EXPIRED':
        case 'PURCHASE_DELAYED':
            custom_log("Evento: Compra com Chargeback, Expirada ou Atrasada", 'result');
            update_subscription_status($conn, $data, 'expirada', true, 0); // Zerar dias_restantes
            break;

        case 'CHARGEBACK':
            custom_log("Evento: Chargeback", 'result');
            update_subscription_status($conn, $data, 'chargeback');
            break;

        case 'ASSINATURA_CANCELADA':
        case 'SUBSCRIPTION_CANCELLATION':
            custom_log("Evento: Cancelamento de Assinatura", 'result');
            update_subscription_status($conn, $data, 'assinatura_cancelada', true, 0, 'Cancelada'); // Define status_assinatura para "Cancelada"
            break;

        default:
            custom_log("Evento desconhecido: $event_type", 'error');
            break;
    }
}

// Função para salvar os dados no banco (tabela usuários e assinaturas)
function save_user_and_subscription($conn, $data, $dias_restantes = 0) {
    try {
        custom_log('Iniciando função de salvar usuário e assinatura.', 'info');

        // Extrair dados necessários
        $user_email = $data['data']['buyer']['email'] ?? null;
        $user_name = $data['data']['buyer']['name'] ?? null;
        $cpf = $data['data']['buyer']['cpf'] ?? '00000000000'; // Coletar CPF com padrão
        $transaction_code = $data['data']['purchase']['transaction'] ?? 'sem_transacao'; // Adiciona o código da transação
        $payment_type = $data['data']['purchase']['payment']['type'] ?? 'desconhecido'; // Tipo de pagamento
        $status_compra = $data['data']['purchase']['status'] ?? 'desconhecido'; // Status da compra
        $status_assinatura = $data['data']['subscription']['status'] ?? 'desconhecido'; // Status da assinatura

        // Verificações básicas
        if (!$user_email || !$user_name) {
            custom_log("Erro: Email ou nome do usuário não encontrados. Dados: " . print_r($data, true), 'error');
            return;
        }

        $username = preg_replace('/\s+/', '_', strtolower($user_name)); // Remove espaços e converte para minúsculas
        $subscription_code = $data['data']['subscription']['subscriber']['code'] ?? null; // Código da assinatura
        $subscription_plan = $data['data']['subscription']['plan']['name'] ?? 'desconhecido';
        $purchase_date = isset($data['data']['purchase']['order_date']) 
                        ? date('Y-m-d H:i:s', $data['data']['purchase']['order_date'] / 1000) 
                        : null;
        $status = 'ativa'; // Status inicial

        custom_log("Dados coletados: Email: $user_email, Nome: $user_name, Código de Assinatura: $subscription_code, Transação: $transaction_code, Tipo de Pagamento: $payment_type, Status Compra: $status_compra, Status Assinatura: $status_assinatura", 'info');

        if (!$subscription_code) {
            custom_log("Erro: Código de assinatura não encontrado. Dados: " . print_r($data, true), 'error');
            return;
        }

        // Verifica se o usuário já existe na tabela "usuarios"
        $stmt = $conn->prepare("SELECT id FROM usuarios WHERE email = ?");
        $stmt->execute([$user_email]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        custom_log("Resultado da busca do usuário no banco: " . print_r($user, true), 'debug');

        $password = password_hash('senha_default', PASSWORD_DEFAULT); // Senha padrão

        if ($user) {
            // Usuário já existe, atualiza os dados
            $id = $user['id'];
            $stmt = $conn->prepare("UPDATE usuarios SET nome = ?, username = ?, cpf = ? WHERE id = ?");
            if ($stmt->execute([$user_name, $username, $cpf, $id])) {
                custom_log("Usuário atualizado: $user_email", 'result');
            } else {
                custom_log("Erro ao atualizar usuário: " . print_r($stmt->errorInfo(), true), 'error');
            }
        } else {
            // Insere um novo usuário
            $stmt = $conn->prepare("INSERT INTO usuarios (username, nome, email, password, cpf) VALUES (?, ?, ?, ?, ?)");
            if ($stmt->execute([$username, $user_name, $user_email, $password, $cpf])) {
                $id = $conn->lastInsertId();
                custom_log("Novo usuário inserido: $user_email", 'result');
            } else {
                custom_log("Erro ao inserir novo usuário: " . print_r($stmt->errorInfo(), true), 'error');
            }
        }

        // Verifica se a assinatura já existe para este usuário
        $stmt = $conn->prepare("SELECT id FROM assinaturas WHERE codigo_assinatura = ?");
        $stmt->execute([$subscription_code]);
        $subscription = $stmt->fetch(PDO::FETCH_ASSOC);
        custom_log("Resultado da busca de assinatura no banco: " . print_r($subscription, true), 'debug');

        if ($subscription) {
            // Assinatura já existe, atualiza os dados
            $stmt = $conn->prepare("UPDATE assinaturas SET plano = ?, status = ?, data_aprovacao = ?, status_assinatura = ?, dias_restantes = ? WHERE codigo_assinatura = ?");
            if ($stmt->execute([$subscription_plan, $status, $purchase_date, $status_assinatura, $dias_restantes, $subscription_code])) {
                custom_log("Assinatura atualizada para: $subscription_code", 'result');
            } else {
                custom_log("Erro ao atualizar assinatura: " . print_r($stmt->errorInfo(), true), 'error');
            }
        } else {
            // Insere nova assinatura
            $stmt = $conn->prepare("INSERT INTO assinaturas (usuario_id, codigo_assinatura, plano, status, data_aprovacao, transacao, tipo_pagamento, status_compra, status_assinatura, dias_restantes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            if ($stmt->execute([$id, $subscription_code, $subscription_plan, $status, $purchase_date, $transaction_code, $payment_type, $status_compra, $status_assinatura, $dias_restantes])) {
                custom_log("Nova assinatura inserida: $subscription_code", 'result');
            } else {
                custom_log("Erro ao inserir nova assinatura: " . print_r($stmt->errorInfo(), true), 'error');
            }
        }

        custom_log("Dados salvos/atualizados com sucesso para o usuário: $user_email", 'result');
    } catch (Exception $e) {
        custom_log("Erro ao salvar dados no banco: " . $e->getMessage(), 'error');
    }
}

// Função para atualizar o status de uma assinatura
function update_subscription_status($conn, $data, $new_status, $reset_days = false, $dias_restantes = 0, $status_assinatura = null) {
    try {
        custom_log("Iniciando atualização de status da assinatura para: $new_status", 'info');

        // Verifica se a assinatura está presente e se o código é válido
        $subscription = $data['data']['subscription']['subscriber']['code'] ?? null;

        if (empty($subscription)) {
            custom_log("Erro: Código de assinatura não encontrado. Dados: " . print_r($data, true), 'error');
            return;
        }

        // Define dias_restantes
        if ($reset_days) {
            $dias_restantes = 0;
        }

        // Atualiza o status e status_assinatura da assinatura no banco
        $stmt = $conn->prepare("UPDATE assinaturas SET status = ?, dias_restantes = ?, status_assinatura = ? WHERE codigo_assinatura = ?");
        if ($stmt->execute([$new_status, $dias_restantes, $status_assinatura, $subscription])) {
            custom_log("Status da assinatura atualizado para: $new_status com dias_restantes = $dias_restantes e status_assinatura = $status_assinatura", 'result');
        } else {
            custom_log("Erro ao atualizar status da assinatura: " . print_r($stmt->errorInfo(), true), 'error');
        }
    } catch (Exception $e) {
        custom_log("Erro ao atualizar o status da assinatura: " . $e->getMessage(), 'error');
    }
}

// Verifica se a requisição é POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    custom_log("Requisição POST recebida.", 'info');
    
    // Pega o JSON enviado pela Hotmart
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    // Verifica se o JSON foi decodificado corretamente
    if ($data) {
        custom_log('Dados recebidos: ' . print_r($data, true), 'result');
        custom_log('Estrutura de dados recebidos: ' . print_r($data, true), 'debug');

        // Identifica o tipo de evento
        $event_type = $data['event'] ?? null;

        // Processa o evento
        if ($event_type) {
            process_event($conn, $event_type, $data);
        } else {
            custom_log("Erro: Evento não especificado.", 'error');
        }

        // Retorna um status de sucesso
        http_response_code(200);
        echo json_encode(['status' => 'success']);
    } else {
        // Se não for um JSON válido
        custom_log("Erro: JSON inválido ou não decodificado corretamente.", 'error');
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Invalid JSON']);
    }
} else {
    // Se não for uma requisição POST
    custom_log("Erro: Requisição não é POST.", 'error');
    http_response_code(405); // Método não permitido
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
}
?>
